<?php

namespace App\Http\Conversations;

use App\Models\ClinicCategory;
use App\Models\Clinic;

use App\Models\Doctor;

use App\Models\Service;
use App\Models\ServiceCategory;
use App\Models\DoctorCategory;
use BotMan\BotMan\Messages\Incoming\Answer;
use BotMan\BotMan\Messages\Outgoing\Question;
use BotMan\BotMan\Messages\Outgoing\Actions\Button;
use BotMan\BotMan\Messages\Conversations\Conversation;
use BotMan\BotMan\Messages\Outgoing\OutgoingMessage;
use Exception;
use Illuminate\Support\Facades\Log;

class ServicesConversations extends Conversation
{
    /**
     * Start the conversation
     *
     * @return mixed
     */

    protected $name = '';
    protected $email = '';
    protected $service = '';
    protected $list_of_keywords = [];

    public function __construct()
    {
        // Load keywords from the external file
        $this->list_of_keywords = require __DIR__ . '/ListOfKeywords.php';
    }

    public function run()
    {
        $this->ask('Hi there! What is your Name?', function (Answer $answer) {
            $this->name = $answer->getText();
            if (empty($this->name)) {
                $this->say('Name cannot be empty.');
                return $this->repeat();
            }
            $this->say('Hi! Nice to meet you, ' . $this->name);
            $this->askEmail();
        });
    }

    public function askEmail()
    {
        $this->ask('What is your Email?', function (Answer $answer) {
            $this->email = $answer->getText();
            if (!filter_var($this->email, FILTER_VALIDATE_EMAIL)) {
                $this->say('Please enter a valid email address.');
                return $this->repeat();
            }
            $this->askForOptions();
        });
    }

    protected function askForOptions()
    {

        $questions = Question::create('Please Select your needy Option?')
            ->fallback('Unable to ask question')->addButtons([
                Button::create('I need a Doctor')->value('doctor'),
                Button::create('I need an Clinic')->value('clinic'),
                Button::create('I need a Service')->value('service'),
            ]);
        $this->ask($questions, function (Answer $answer) {
            if (!$answer->isInteractiveMessageReply()) {
                $this->say('Please select an option.');
                return $this->repeat();
            }
            $selected_option = $answer->getValue();
            $this->say('You selected : ' . $selected_option);
            if ($selected_option == 'doctor') {
                $this->askDoctorCategories();
            } elseif ($selected_option == 'clinic') {
                $this->askClinicCategories();
            } else {
                $this->askServiceCategories();
            }
        });
    }

    protected function askDoctorCategories()
    {
        try {
            $doctor_categories = DoctorCategory::active()->get();
            if ($doctor_categories->isEmpty()) {
                $this->say('No doctor categories available at the moment.');
                return;
            }

            $questions = Question::create('Which type of doctor do you want?')
                ->fallback('Unable to ask question');

            foreach ($doctor_categories as $doctor_category) {
                $questions->addButton(Button::create($doctor_category->name)->value($doctor_category->id));
            }

            $this->ask($questions, function (Answer $answer) {
                if (!$answer->isInteractiveMessageReply()) {
                    $this->say('Please select a doctor category.');
                    return $this->repeat();
                }

                $selectedCategoryId = $answer->getValue();
                $this->say('You selected category ID: ' . $selectedCategoryId);
                $this->doctorSuggestions($selectedCategoryId, 'category');
            });
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving service categories. Please try again later.');
            Log::error('Error fetching service categories: ' . $e->getMessage());
        }
    }


    protected function askClinicCategories()
    {
        try {
            $clinic_categories = ClinicCategory::active()->get();
            if ($clinic_categories->isEmpty()) {
                $this->say('No clinic categories available at the moment.');
                return;
            }

            $questions = Question::create('Which type of clinic do you want?')
                ->fallback('Unable to ask question');

            foreach ($clinic_categories as $doctor_category) {
                $questions->addButton(Button::create($doctor_category->name)->value($doctor_category->id));
            }

            $this->ask($questions, function (Answer $answer) {
                if (!$answer->isInteractiveMessageReply()) {
                    $this->say('Please select a clinic category.');
                    return $this->repeat();
                }

                $selectedCategoryId = $answer->getValue();
                $this->say('You selected category ID: ' . $selectedCategoryId);
                $this->clinicSuggestions($selectedCategoryId);

            });
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving service categories. Please try again later.');
            Log::error('Error fetching service categories: ' . $e->getMessage());
        }
    }





    protected function askServiceCategories()
    {
        try {
            $service_categories = ServiceCategory::all();
            if ($service_categories->isEmpty()) {
                $this->say('No service categories available at the moment.');
                return;
            }

            $questions = Question::create('Which type of service do you want?')
                ->fallback('Unable to ask question');

            foreach ($service_categories as $service_category) {
                $questions->addButton(Button::create($service_category->name)->value($service_category->id));
            }

            $this->ask($questions, function (Answer $answer) {
                if (!$answer->isInteractiveMessageReply()) {
                    $this->say('Please select a service category.');
                    return $this->repeat();
                }

                $selectedCategoryId = $answer->getValue();
                $this->say('You selected category ID: ' . $selectedCategoryId);
                $this->askService($selectedCategoryId);
            });
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving service categories. Please try again later.');
            Log::error('Error fetching service categories: ' . $e->getMessage());
        }
    }

    protected function askService($selectedCategoryId)
    {
        try {
            $services = Service::where('service_category_id', $selectedCategoryId)->get();
            if ($services->isEmpty()) {
                $this->say('No services available for this category.');
                return;
            }

            $questions = Question::create('List of Services');

            foreach ($services as $service) {
                $questions->addButton(Button::create($service->name)->value($service->id));
            }

            $this->ask($questions, function (Answer $answer) {
                if (!$answer->isInteractiveMessageReply()) {
                    $this->say('Please select a service.');
                    return $this->repeat();
                }

                $service_id = $answer->getValue();
                $this->say('You selected Service ID: ' . $service_id);
                $this->doctorSuggestions($service_id, 'service');
            });
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving services. Please try again later.');
            Log::error('Error fetching services: ' . $e->getMessage());
        }
    }

    protected function doctorSuggestions($id, $type)
    {
        try {
            $doctors = Doctor::limit(5);
            if ($type == 'category') {
                $doctors->whereHas('doctor_categories', function ($query) use ($id) {
                    $query->where('doctor_category_id', $id);
                });
            } else {
                $doctors = $doctors->has('doctor_services');
            }
            $doctors = $doctors->get();

            if ($doctors->isEmpty()) {
                $this->say('No doctors found for this service.');
                return;
            }


            $listItems = '';
            $this->say('list of Doctors: ');
            foreach ($doctors as $doctor) {
                $listItems .= '<li>Name: <a href="' . url('/doctor/profile/' . $doctor->user_name) . '" target="_blank">' . $doctor->name . '</a></li>';
            }
            $messageContent = '<ul>' . $listItems . '</ul>';
            $message = OutgoingMessage::create($messageContent);
            $this->say($message);
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving doctors. Please try again later.');
            Log::error('Error fetching doctors: ' . $e->getMessage());
        }
    }

    protected function clinicSuggestions($id)
    {
        try {
            $clinics = Clinic::limit(5);
            $clinics->whereHas('clinic_categories', function ($query) use ($id) {
                $query->where('clinic_id', $id);
            });

            $clinics = $clinics->get();

            if ($clinics->isEmpty()) {
                $this->say('No clinics found for this service.');
                return;
            }

            $listItems = '';
            $this->say('list of clinics: ');
            foreach ($clinics as $clinic) {
                $listItems .= '<li>Name: <a href="' . url('/clinic/profile/' . $clinic->user_name) . '" target="_blank">' . $clinic->name . '</a></li>';
            }
            $messageContent = '<ul>' . $listItems . '</ul>';
            $message = OutgoingMessage::create($messageContent);
            $this->say($message);
        } catch (Exception $e) {
            $this->say('An error occurred while retrieving clinics. Please try again later.');
            Log::error('Error fetching clinics: ' . $e->getMessage());
        }
    }
}
